"use client"

import React, { useEffect } from "react"
import { MapContainer, TileLayer, useMap } from "react-leaflet"

import "leaflet/dist/leaflet.css"
import "leaflet.markercluster/dist/MarkerCluster.css"
import "leaflet.markercluster/dist/MarkerCluster.Default.css"
import "leaflet.markercluster"

import L, { LatLngExpression } from "leaflet"
import { TODO } from "@/types/utils"
import { MapDataPoint } from "../types/map-data-point"

interface MapProps {
  data: MapDataPoint[]
  showOrigin: boolean
  showDestination: boolean
}

const michiganCenter: LatLngExpression = [44.3148, -85.6024]

const getColorByClusterSize = (count: number, isOrigin: boolean): string => {
  const themeColors = isOrigin
    ? [
      "rgba(47,210,77,0.7)",
      "rgba(34,179,60,0.7)",
      "rgba(20,148,43,0.7)",
      "rgba(9,115,29,0.7)",
    ]
    : [
      "rgba(240,39,39,0.7)",
      "rgba(205,34,34,0.7)",
      "rgba(170,23,23,0.7)",
      "rgba(133,11,11,0.7)",
    ]

  if (count > 50) return themeColors[3]
  if (count > 20) return themeColors[2]
  if (count > 10) return themeColors[1]
  return themeColors[0]
}

const createClusterLayer = (
  map: L.Map,
  data: MapDataPoint[],
  isOrigin: boolean
) => {
  const markers = L.markerClusterGroup({
    iconCreateFunction: function (cluster: TODO) {
      const count = cluster.getChildCount()
      const color = getColorByClusterSize(count, isOrigin)
      return L.divIcon({
        html: `<div style="background-color:${color};border-radius:50%;width:40px;height:40px;display:flex;align-items:center;justify-content:center;color:white;font-weight:bold;">${count}</div>`,
        className: "custom-cluster-icon",
      })
    },
  })

  data.forEach((item) => {
    const position = isOrigin ? item.from : item.to
    if (position) {
      const marker = L.marker(position, {
        icon: L.divIcon({
          className: "",
          html: `<div style="background-color:${isOrigin ? "#30c469" : "#f03030"
            };border-radius:50%;border-width:2px;border-color:gray;width:10px;height:10px;"></div>`,
        }),
      }).bindPopup(
        `<b>${isOrigin ? "Origin" : "Destination"}</b><br/>Position: ${position}`
      )
      markers.addLayer(marker)
    }
  })
  map.addLayer(markers)
  return markers
}

const ClusterLayer = ({
  data,
  showOrigin,
  showDestination,
}: {
    data: MapDataPoint[]
    showOrigin: boolean
    showDestination: boolean
}) => {
  const map = useMap()

  useEffect(() => {
    let originLayer: L.LayerGroup | undefined
    let destinationLayer: L.LayerGroup | undefined

    if (showOrigin) {
      originLayer = createClusterLayer(map, data, true)
    }

    if (showDestination) {
      destinationLayer = createClusterLayer(map, data, false)
    }

    return () => {
      if (originLayer) map.removeLayer(originLayer)
      if (destinationLayer) map.removeLayer(destinationLayer)
    }
  }, [map, data, showOrigin, showDestination])

  return null
}

const Map: React.FC<MapProps> = ({ data, showOrigin, showDestination }) => {
  return (
    <MapContainer
      id="leaflet-map"
      center={michiganCenter}
      zoom={8}
      scrollWheelZoom={true}
      style={{ height: "100%", width: "100%" }}
      className="rounded-lg shadow-md z-50"
    >
      <TileLayer
        url="https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png"
        attribution='&copy; <a href="https://www.openstreetmap.org/copyright">OpenStreetMap</a> contributors'
      />
      <ClusterLayer
        data={data}
        showOrigin={showOrigin}
        showDestination={showDestination}
      />
    </MapContainer>
  )
}

export default Map
